<?php
session_start();
require_once '../config/db.php';
header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit();
}

// Get M-Pesa settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
$stmt->execute();
$mpesaSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Check if M-Pesa is enabled and all required credentials are set
if (($mpesaSettings['enable_mpesa'] ?? '0') !== '1') {
    echo json_encode(['success' => false, 'message' => 'M-Pesa payments are currently disabled.']);
    exit();
}

// Validate required credentials
$requiredCredentials = ['mpesa_consumer_key', 'mpesa_consumer_secret', 'mpesa_shortcode', 'mpesa_passkey'];
foreach ($requiredCredentials as $credential) {
    if (empty($mpesaSettings[$credential])) {
        echo json_encode(['success' => false, 'message' => 'M-Pesa API credentials are not properly configured.']);
        exit();
    }
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);
$amount = floatval($data['amount'] ?? 0);
$phone = preg_replace('/\D/', '', $data['phone'] ?? '');
$customerId = $data['customer_id'] ?? null;
$customerName = $data['customer_name'] ?? '';
$invoiceId = $data['invoice_id'] ?? null;

if ($amount <= 0 || strlen($phone) < 10) {
    echo json_encode(['success' => false, 'message' => 'Invalid amount or phone number.']);
    exit();
}

// Validate invoice exists and is unpaid
if ($invoiceId) {
    $stmt = $pdo->prepare("SELECT * FROM invoices WHERE id = ? AND status != 'paid'");
    $stmt->execute([$invoiceId]);
    $invoice = $stmt->fetch();
    
    if (!$invoice) {
        echo json_encode(['success' => false, 'message' => 'Invoice not found or already paid.']);
        exit();
    }
    
    // Allow partial payments - no need to validate amount matches total
    if ($amount > $invoice['total_amount']) {
        echo json_encode(['success' => false, 'message' => 'Payment amount cannot exceed invoice total.']);
        exit();
    }
}

// Format phone number to 254 format if not already
if (!preg_match('/^254/', $phone)) {
    $phone = '254' . ltrim($phone, '0');
}

// Daraja API credentials
$consumerKey = $mpesaSettings['mpesa_consumer_key'];
$consumerSecret = $mpesaSettings['mpesa_consumer_secret'];
$shortcode = $mpesaSettings['mpesa_shortcode'];
$passkey = $mpesaSettings['mpesa_passkey'];
$env = $mpesaSettings['mpesa_env'] ?? 'sandbox';

// Daraja API URLs
$baseUrl = ($env === 'production') ? 'https://api.safaricom.co.ke' : 'https://sandbox.safaricom.co.ke';

try {
    // 1. Get access token
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . '/oauth/v1/generate?grant_type=client_credentials');
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Authorization: Basic ' . base64_encode($consumerKey . ':' . $consumerSecret)]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $curlError = curl_error($ch);
        curl_close($ch);
        throw new Exception('Failed to get access token: ' . $curlError);
    }
    
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON response from M-Pesa API: ' . json_last_error_msg());
    }
    
    if (isset($result['errorCode'])) {
        throw new Exception('M-Pesa API Error: ' . ($result['errorMessage'] ?? 'Unknown error') . ' (Code: ' . $result['errorCode'] . ')');
    }
    
    if (!isset($result['access_token'])) {
        throw new Exception('Invalid access token response: ' . ($result['error_description'] ?? 'No access token in response'));
    }
    
    $accessToken = $result['access_token'];

    // 2. Initiate STK Push
    $timestamp = date('YmdHis');
    $password = base64_encode($shortcode . $passkey . $timestamp);
    
    $stkPushData = [
        'BusinessShortCode' => $shortcode,
        'Password' => $password,
        'Timestamp' => $timestamp,
        'TransactionType' => 'CustomerPayBillOnline',
        'Amount' => $amount,
        'PartyA' => $phone,
        'PartyB' => $shortcode,
        'PhoneNumber' => $phone,
        'CallBackURL' => 'https://' . $_SERVER['HTTP_HOST'] . '/api/mpesa_invoice_callback.php',
        'AccountReference' => $invoiceId ? "INV{$invoiceId}" : ($customerId ? "CUST{$customerId}" : 'POS'),
        'TransactionDesc' => $invoiceId ? "Invoice Payment INV{$invoiceId}" : ($customerName ? "Payment by {$customerName}" : 'POS Payment')
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . '/mpesa/stkpush/v1/processrequest');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($stkPushData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        throw new Exception('Failed to initiate STK Push: ' . curl_error($ch));
    }
    
    $result = json_decode($response, true);
    curl_close($ch);

    if (isset($result['ResponseCode']) && $result['ResponseCode'] === '0') {
        // Log the STK Push request
        $stmt = $pdo->prepare("INSERT INTO mpesa_transactions (phone, amount, customer_id, customer_name, merchant_request_id, checkout_request_id, response_code, response_description, invoice_id, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        $stmt->execute([
            $phone,
            $amount,
            $customerId,
            $customerName,
            $result['MerchantRequestID'] ?? null,
            $result['CheckoutRequestID'] ?? null,
            $result['ResponseCode'] ?? null,
            $result['ResponseDescription'] ?? null,
            $invoiceId
        ]);

        // Create a pending payment record for the invoice
        if ($invoiceId) {
            $stmt = $pdo->prepare("
                INSERT INTO payments (
                    invoice_id, payment_method, payment_date, amount, 
                    reference_number, notes, processed_by, status, created_at
                ) VALUES (?, 'mpesa', CURDATE(), ?, ?, ?, ?, 'pending', NOW())
            ");
            
            $stmt->execute([
                $invoiceId,
                $amount,
                'MPESA-' . ($result['CheckoutRequestID'] ?? ''),
                'M-Pesa STK Push initiated - CheckoutRequestID: ' . ($result['CheckoutRequestID'] ?? ''),
                $_SESSION['user_id']
            ]);
        }

        echo json_encode([
            'success' => true,
            'message' => 'STK Push sent successfully. Please check your phone to complete the payment.',
            'checkout_request_id' => $result['CheckoutRequestID'] ?? null,
            'merchant_request_id' => $result['MerchantRequestID'] ?? null
        ]);
    } else {
        throw new Exception('STK Push failed: ' . ($result['ResponseDescription'] ?? 'Unknown error'));
    }

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ]);
} 